/* --------------------------------------------------------------
 product_image_list.js 2020-01-23
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Product Images List Controller
 *
 * This controller handles the images sort
 *
 * @module Controllers/product_image_list
 */

gx.controllers.module(
  // ------------------------------------------------------------------------
  // CONTROLLER NAME
  // ------------------------------------------------------------------------
  'product_image_list',

  // ------------------------------------------------------------------------
  // CONTROLLER LIBRARIES
  // ------------------------------------------------------------------------
  [
    `${jse.source}/vendor/jquery-ui-dist/jquery-ui.min.css`,
    `${jse.source}/vendor/jquery-ui-dist/jquery-ui.js`,
    'loading_spinner'
  ],

  // ------------------------------------------------------------------------
  // CONTROLLER BUSINESS LOGIC
  // ------------------------------------------------------------------------
  function (data) {
    'use strict'

    const module = {}

    /**
     * Edit modal element
     *
     * @type {jQuery|HTMLElement}
     */
    const $editModal = $('.edit-panel')

    /**
     * Edit collection modal element
     *
     * @type {jQuery|HTMLElement}
     */
    const $editCollectionModal = $('.edit-collection-panel')

    /**
     * Confirm modal element
     *
     * @type {jQuery|HTMLElement}
     */
    const $confirmModal = $('.confirm-modal-panel')

    /**
     * Toast message element
     *
     * @type {jQuery|HTMLElement}
     */
    const $toastMessage = $('.request-status-wrapper')

    /**
     * Selected image collection value
     */
    const initialSelectedList = data.selectedList

    /**
     *
     * @type {*}
     */
    const appUrl = jse.core.config.get('appUrl')

    /**
     * Load spinner instance
     */
    let loadingSpinner = null

    /**
     *
     */
    let toastTimeoutID = null

    /**
     * Default options
     *
     * @type {{sortable: {cursor: string, containment: string, sortableList: string, handle: string, placeholder: string, axis: string, opacity: number, items: string}, filemanager: {subFolder: string, top: number, left: number, width: number, id: string, lang: string, height: number}}}
     */
    const defaults = {
      sortable: {
        items: 'li.row',
        axis: 'y',
        cursor: 'move',
        handle: '.sort-handle',
        containment: 'document',
        opacity: 0.75,
        placeholder: 'col-md-12 list-element sort-placeholder',
        sortableList: 'ul.configuration-image-list'
      },
      filemanager: {
        id: 'add-image-collection',
        subFolder: 'images/product_images/original_images',
        popup: 1,
        lang: 'de',
        width: 800,
        height: 600,
        top: 100,
        left: 100,
        useFileManager: false
      }
    }

    /**
     * Sortable options
     */
    const sortableOptions = $.extend(true, {}, defaults.sortable, data)

    /**
     * File manager options
     */
    const filemanagerOptions = $.extend(true, {}, defaults.filemanager, data)

    let fileManager = null

    /**
     * Sortable list element.
     *
     * @type {jQuery}
     */
    const $sortableList = $(sortableOptions.sortableList)

    /**
     * Handler when updating the order of the image list
     */
    const handleOnSortUpdate = async () => {
      loading(true)

      try {
        const newOrder = $sortableList
          .sortable('toArray', { attribute: 'data-list-element-id' })
          .reduce((newValue, currentValue, currentIndex) => {
            newValue.push({
              imageId: parseInt(currentValue),
              sortIndex: currentIndex
            })
            return newValue
          }, [])

        const result = await updateImagesSort({ sort: JSON.stringify(newOrder) })

        if (result.success) {
          toastSuccess(result.message)
        } else {
          toastError(result.message)
        }

      } catch (e) {
        toastError(e.message)
      } finally {
        loading(false)
      }
    }

    /**
     * Event listener for add image button
     */
    const addImageButtonEventListener = () => {
      $('.add-image').click(function () {
        const { id, subFolder, lang, width, height, top, left, popup, useFileManager } = filemanagerOptions

        if (!useFileManager) {
          toastError(jse.core.lang.translate('RESPONSIVE_FILEMANAGER_REQUIRED', 'product_image_lists'))
          return;
        }

        const urlParams = [
          `field_id=${id}`,
          `crossdomain=1`,
          `sub_folder=${subFolder}`,
          `lang=${lang}`,
          `popup=${popup}`
        ].join('&');

        fileManager = parent.window.open(
          `${appUrl}/ResponsiveFilemanager/filemanager/dialog.php?${urlParams}`,
          'ResponsiveFilemanager',
          `scrollbars=1,width=${width},height=${height},top=${top},left=${left}`
        )
      })
    }

    const addResponsiveFileManagerEventListener = () =>{
      const onMessage = async event => {
        const { data } = event,
          { sender, url, field_id } = data

        if (sender !== 'responsivefilemanager' || field_id !== filemanagerOptions.id) {
          return
        }

        const correctUrl = url.replace(/([^:]\/)\/+/g, '$1')

        await handleAddImageToCollection({
          url: correctUrl,
          localPath: normalizeLocalPath(correctUrl)
        })

        fileManager.close()
        fileManager = null
      }

      window.addEventListener('message', onMessage, false)
    }

    /**
     * Removes the origin domain from the image path
     *
     * @param url
     * @returns {*}
     */
    const normalizeLocalPath = url => {
      const regex = `${appUrl}(\/)?`

      return url.replace(new RegExp(regex, 'g'), '')
    }

    /**
     * Event listener for creating a new image collection
     */
    const createNewCollectionButtonEventListener = () => {
      $('#create-new-collection').on('click', function () {
        clearCreateNewCollectionInput()

        $('.selected-collection-wrapper').addClass('create-new-collection')
        $('.select-collection').hide()
        $('.create-collection').show()
        $('.create-collection input').focus()

        if (hasCollection()) {
          $('#select-collection').show()
        }

        $(this).hide()
      })
    }

    /**
     * Event listener for clicking on "Select collection" text/button
     */
    const selectCollectionButtonEventListener = () => {
      $('#select-collection').on('click', function () {
        $('.selected-collection-wrapper').removeClass('create-new-collection')
        $('.create-collection').hide()
        $('.select-collection').show()
        $('#create-new-collection').show()
        $(this).hide()
      })
    }

    /**
     * Event listener for opening the Edit image modal
     */
    const openEditImageModalEventListener = () => {
      $(document).on('click', '.edit-image', function () {
        const $parent = $(this).parents('.collection-image-wrapper'),
              data = $parent.data('image')

        handleEditImageModal({
          id: data.id,
          src: data.webFilePath,
          localPath: normalizeLocalPath(data.webFilePath),
          titles: data.titles,
          altTitles: data.altTitles
        })
      })
    }

    /**
     * Event listener for closing Edit image modal
     */
    const closeEditModalButtonEventListener = () => {
      $('.edit-panel .edit-modal .close-btn').on('click', handleCloseEditImageModal)
    }

    /**
     * Event listener for canceling/closing the Edit image modal
     */
    const cancelEditModalEventListener = () => {
      $(document).on('click', '.edit-modal-cancel', handleCloseEditImageModal)
    }

    /**
     * Event listener for hitting the save button inside Edit Image Modal
     */
    const saveEditModalEventListener = () => {
      $editModal.on('click', '.edit-modal-save', handleSaveImageModal)
    }

    /**
     * Event listener for closing Edit collection modal
     */
    const closeEditCollectionModalButtonEventListener = () => {
      $('.edit-collection-panel .edit-modal .close-btn').on('click', handleCloseEditCollectionModal)
    }

    /**
     * Event listener for opening the Edit collection modal
     */
    const openEditCollectionModalEventListener = () => {
      $(document).on('click', '.edit-collection', handleEditCollectionModal)
    }

    /**
     * Event listener for canceling/closing the Edit image modal
     */
    const cancelEditCollectionModalButtonEventListener = () => {
      $(document).on('click', '.edit-collection-modal-cancel', handleCloseEditCollectionModal)
    }

    /**
     * Listener for clicking on save button inside Edit Collection Modal
     */
    const updateEditCollectionModalButtonEventListener = () => {
      $editCollectionModal.on('click', '.edit-collection-modal-save', handleUpdateCollectionModal)
    }

    /**
     * Handler for closing Edit Image Modal
     */
    const handleCloseEditCollectionModal = () => {
      $editCollectionModal.fadeOut(() => clearModalInputs($editCollectionModal))
    }

    /**
     * Event listener for deleting image button
     */
    const deleteImageButtonEventListener = () => {
      $(document).on('click', '.delete-image', function () {
        handleDeleteImage($(this).parents('.collection-image-wrapper'))
      })
    }

    /**
     * Event listener when changing the image collection dropdown
     */
    const imageCollectionOnChangeEventListener = () => {
      $('#combi_image_collection').change(function () {
        const val = parseInt($(this).val())
        handleOnChangeImageCollection(val)
      })
    }

    /**
     * Event listener for creating new collection button
     */
    const createCollectionButtonEventListener = () => {
      $('button.create-new-collection').on('click', async () => {
        const $input = $('input[name=new-collection-name]'),
          name = $input.val()

        if (!name) {
          $input.addClass('error')

          return false
        }

        await handleCreateNewCollection(name)

      })
    }

    /**
     * Event listener for deleting collection list button
     */
    const deleteCollectionButtonEventListener = () => {
      $('button.delete-collection').on('click', async () => {
        handleDeleteCollection($('#combi_image_collection option:selected').val())
      })
    }

    /**
     * Prevents submit the modal forms by pressing enter.
     * It triggers the save button instead of submitting it.
     *
     */
    const preventSubmitFormModals = () => {
      $editModal.find('#edit-image-form').submit(e => {
        $editModal.find('.edit-modal-save').trigger('click')
        return false
      })
      $editCollectionModal.find('#edit-collection-form').submit(e => {
        $editCollectionModal.find('.edit-collection-modal-save').trigger('click')
        return false
      })
    }

    /**
     * Handler for closing Edit Image Modal
     */
    const handleCloseEditImageModal = () => {
      if ($editModal.find('input[name=id]').val() === '0') {
        removeLastImageElement()
      }

      $editModal.fadeOut(() => clearModalInputs($editModal))
    }

    /**
     * Handler to add an image to a collection
     *
     * @returns {void}
     */
    const handleSaveImageModal = async () => {
      if (!isValidEditModalData($editModal)) {
        return
      }

      loading(true)

      try {
        let formData = parseEditModalData($('form#edit-image-form').serializeArray()),
            result = null

        if (formData.id !== '0') {
          formData.imageId = formData.id
          delete formData.id
          result = await updateImageTexts(formData)
        } else {
          delete formData.id
          result = await addImageToCollection(formData)
        }

        resetImagesWrapper()
        loadImagesCollection(getSelectedList().id)
        $editModal.fadeOut(() => clearModalInputs($editModal))

        const {success, message = null} = result

        if (success) {
          toastSuccess(message || jse.core.lang.translate('MODAL_SUCCESS_TEXT', 'product_image_lists'))
        } else {
          toastError(message || jse.core.lang.translate('MODAL_ERROR_TEXT', 'product_image_lists'))
        }

      } catch (e) {
        toastError(e.message)
      } finally {
        loading(false)
      }
    }

    /**
     * Handler to update the collection's name
     *
     * @returns {void}
     */
    const handleUpdateCollectionModal = async () => {
      if (!isValidEditModalData($editCollectionModal)) {
        return
      }

      loading(true)
      try {
        let formData = parseEditModalData($('form#edit-collection-form').serializeArray())

        const { id, name } = formData
        const {success, message} = await updateCollectionName({ listId: id, listName: name })

        if (success) {
          toastSuccess(message)
        } else {
          toastError(message)
        }

        loadCollections(id)
        $editCollectionModal.fadeOut()
      } catch (e) {
        toastError(e.message)
      } finally {
        loading(false)
      }
    }

    /**
     * Validates edit modals
     *
     * @param $modal
     * @returns {boolean}
     */
    const isValidEditModalData = $modal => {
      $modal.find('form input').removeClass('error')

      const $form = $modal.find('form')
      let errors = 0

      $form
        .find('[class*=required]')
        .each((index, element) => {
          if ($(element).val() === '') {
            $(element).addClass('error')
            errors++
          }
        })

      return errors === 0
    }

    /**
     * Clears the inputs for the given modal element
     *
     * @param $modal
     */
    const clearModalInputs = $modal => {
      $modal.find('input').each((index, element) => {
        $(element)
          .val('')
          .removeClass('error')
      })
    }

    /**
     * Parse edit modal form data to send to back end
     *
     * @param formData
     * @returns {{*}}
     */
    const parseEditModalData = formData => {
      const parsedData = { titles: [], altTitles: [] }

      formData.forEach(element => {
        if (/\[\w+\]/gi.test(element.name)) {
          const key = element.name.replace(/.*\[(\w+)\]/, '$1'),
            value = element.name.replace(/(.*)\[\w+\]/, '$1')

          parsedData[value].push({
            value: element.value,
            languageCode: key
          })
        } else {
          parsedData[element.name] = element.value
        }
      })

      parsedData.titles = JSON.stringify(parsedData.titles)
      parsedData.altTitles = JSON.stringify(parsedData.altTitles)

      return parsedData
    }

    const openConfirmModal = () => {
      return new Promise(resolve => {
        $confirmModal.show()

        $confirmModal.on('click', '.confirm-modal-cancel, .confirm-modal .close-btn', () => {
          resolve(false)
          $confirmModal.hide()
        })
        $confirmModal.find('.confirm-modal-confirm').click(() => {
          resolve(true)
          $confirmModal.hide()
        })
      })
    }

    /**
     * Handler when deleting a image from the list
     * @param $imageWrapper
     */
    const handleDeleteImage = async $imageWrapper => {
      const canProceed = await openConfirmModal()

      if (!canProceed) {
        return
      }

      loading(true)

      try {
        const imageId = parseInt($imageWrapper.data('image').id)
        let message = jse.core.lang.translate('MODAL_SUCCESS_TEXT', 'product_image_lists')

        if (imageId) {
          const result = await deleteImageById(imageId)
          message = result.message
        }

        $imageWrapper.fadeOut(function () {
          $(this).remove()

          if (!hasImage()) {
            setToNoImagesDisplay()
          }
        })

        toastSuccess(message)
      } catch (e) {
        toastError(e.message)
      } finally {
        loading(false)
      }
    }

    /**
     * Clears the new collection input text
     *
     * @returns {void}
     */
    const clearCreateNewCollectionInput = () => {
      $('input[name=new-collection-name]')
        .removeClass('error')
        .val('')
    }

    /**
     * Appends an image to the image wrapper
     *
     * @param $imageBlock
     */
    const appendImageToCollection = $imageBlock => {
      $('.selected-collection-wrapper').removeClass('create-new-collection no-image-selected')
      $('.collection-images-wrapper').css('display', 'flex').append($imageBlock)
    }

    /**
     * Resets/clears the image wrapper
     *
     * @returns {jQuery}
     */
    const resetImagesWrapper = () => $('.collection-images-wrapper').html('')

    /**
     * Replaces the collection dropdown with new collection list
     *
     * @param collections
     * @param selected
     */
    const replaceCollections = (collections, selected = null) => {
      const $collection = $('select#combi_image_collection')
      $collection.html('')
        .append($('<option/>').val(0).html(''))

      collections.forEach(element => {
        const $option = $('<option/>').val(element.id).html(element.name)

        if (selected !== null && parseInt(selected) === element.id) {
          $option.prop('selected', true)
        }

        $collection.append($option)
      })
    }

    /**
     * Initialize the collections list
     *
     * @returns {void}
     */
    const loadCollections = async (selected = null) => {
      const collections = await getCollection()

      replaceCollections(collections, selected)

      if (!collections.length) {
        setToNoImagesDisplay()
        setToNoCollectionsDisplay()
        return
      }

      enableImageSelection()
    }

    /**
     * Initialize the images collection by a given collection ID
     *
     * @param collection_id
     * @returns {void}
     */
    const loadImagesCollection = async collection_id => {
      const images = await getImageCollection(collection_id)

      if (!images.length) {
        setToNoImagesDisplay()
        return
      }
      // $('.selected-collection-wrapper').removeClass('disabled-image-selection')

      images
        .sort((a, b) => a.sortOrder - b.sortOrder)
        .forEach(element => {
          const $imageBlock = $(imageBlockTemplate({ id: element.id, src: element.webFilePath }))

          $imageBlock.data('image', element)
          appendImageToCollection($imageBlock)
        })
    }

    /**
     * Hides the collection image wrapper element
     */
    const setToNoImagesDisplay = () =>  {
      $('.selected-collection-wrapper').addClass('no-image-selected')
      $('div.collection-images-wrapper').hide()
    }

    /**
     * Hide elements when we don't have collections
     */
    const setToNoCollectionsDisplay = () => {
      $('button#select-collection').hide()
      $('button#create-new-collection').trigger('click')
    }

    /**
     * Disables the image placeholder and the edit/delete buttons of the image collection
     */
    const disableImageSelection = () => {
      $('.selected-collection-wrapper').addClass('disabled-image-selection')
      $('button.edit-collection, button.delete-collection').attr('disabled', true)
    }

    /**
     * Enables the image placeholder and the edit/delete buttons of the image collection
     */
    const enableImageSelection = () => {
      $('.selected-collection-wrapper').removeClass('disabled-image-selection')
      $('button.edit-collection, button.delete-collection').removeAttr('disabled')
    }

    /**
     * Returns true if there is at least one collection
     * We check if it is bigger than 1 because the first element it's a blank option
     *
     * @returns {boolean}
     */
    const hasCollection = () => $('#combi_image_collection option').length > 1

    /**
     * Checks if the images wrapper has images
     *
     * @returns {boolean}
     */
    const hasImage = () => $('.collection-images-wrapper > .collection-image-wrapper').length > 0

    /**
     * Removes the last image element from the images container
     */
    const removeLastImageElement = () => {
      // remove last image
      const $lastImageElement = $('.collection-images-wrapper > .collection-image-wrapper:last-child')
      $lastImageElement.remove()

      if (!hasImage()) {
        setToNoImagesDisplay()
      }
    }

    /**
     * Request to get all the image collections
     *
     * @returns {Promise}
     */
    const getCollection = () => {
      const url = [
        appUrl,
        '/admin/admin.php?do=ProductImageListReadAjax'
      ].join('')

      return $.get(url, response => response)
    }

    /**
     * Get all images from a given collection
     *
     * @param collection_id
     * @returns {Promise}
     */
    const getImageCollection = collection_id => {
      const url = [
        appUrl,
        '/admin/admin.php?do=ProductImageListReadAjax/list',
        `&id=${collection_id}`
      ].join('')

      return $.get(url)
        .then(response => response[0].images || [])
    }

    /**
     * Request to create a new collection list
     *
     * @param listName
     */
    const createCollection = listName => {
      const url = [
        appUrl,
        '/admin/admin.php?do=ProductImageListCreateAjax/imageList'
      ].join('')

      return $.post(
        url,
        { listName },
        response => response
      )
    }

    /**
     * Request to delete a collection list
     *
     * @param id
     * @param modifierId
     * @param modifierType
     * @returns {Promise}
     */
    const deleteCollection = ({id, modifierId, modifierType}) => {
      const url = [
        appUrl,
        '/admin/admin.php?do=ProductImageListDeleteAjax/deleteImageListById',
        `&id=${id}&modifierId=${modifierId}&modifierType=${modifierType}`
      ].join('')

      return $.ajax({
        url: url,
        type: 'DELETE'
      }).then(response => response)
    }

    /**
     * Request to add an image to a collection list
     *
     * @param parsedData
     * @returns {*}
     */
    const addImageToCollection = parsedData => {
      const url = [
        appUrl,
        '/admin/admin.php?do=ProductImageListCreateAjax/image'
      ].join('')

      return $.post(
        url,
        parsedData,
        response => response,
        'json'
      )
    }

    /**
     * Updates the text and alt text for the image
     *
     * @param parsedData
     * @returns {*}
     */
    const updateImageTexts = parsedData => {
      const url = [
        appUrl,
        '/admin/admin.php?do=ProductImageListUpdateAjax/updateImageText'
      ].join('')

      return $.post(
        url,
        parsedData,
        response => response,
        'json'
      )
    }

    /**
     * Request to update a collection name
     *
     * @param parsedData
     * @returns {*}
     */
    const updateCollectionName = parsedData => {
      const url = [
        appUrl,
        '/admin/admin.php?do=ProductImageListUpdateAjax/updateImageListName'
      ].join('')

      return $.post(
        url,
        parsedData,
        response => response,
        'json'
      )
    }

    /**
     * Deletes an image from the current image collection
     *
     * @param id
     * @returns {Promise}
     */
    const deleteImageById = id => {
      const url = [
        appUrl,
        `/admin/admin.php?do=ProductImageListDeleteAjax/deleteImageById&id=${id}`
      ].join('')

      return $.ajax({
        url: url,
        type: 'DELETE'
      }).then(response => response)
    }

    /**
     * Updates the order of the images
     *
     * @param parsedData
     * @returns {*}
     */
    const updateImagesSort = parsedData => {
      const url = [
        appUrl,
        '/admin/admin.php?do=ProductImageListUpdateAjax/updateImagesSort'
      ].join('')

      return $.post(
        url,
        parsedData,
        response => response,
        'json'
      )
    }

    /**
     * Handler to create a new collection button event
     *
     * @param name
     * @returns {Promise<void>}
     */
    const handleCreateNewCollection = async name => {
      loading(true)
      try {
        // Ajax request to create a new
        const result = await createCollection(name)

        // Get image collection list
        resetImagesWrapper()
        await loadCollections()
        await loadImagesCollection(
          $('select#combi_image_collection option:last-child')
            .prop('selected', true)
            .val()
        )

        // Trigger "Select collection" button
        $('#select-collection').trigger('click')

        if (result.success) {
          toastSuccess(result.message)
        } else {
          toastError(result.message)
        }

      } catch (e) {
        toastError(e.message)
      } finally {
        loading(false)
      }
    }

    /**
     * Handler for deleting collection button
     *
     * @param id
     * @returns {void}
     */
    const handleDeleteCollection = async id => {
      const canProceed = await openConfirmModal()

      if (!canProceed) {
        return
      }

      loading(true)

      try {
        const modifierId = $('input:hidden[name=modifierId]').val(),
              modifierType = $('input:hidden[name=modifierType]').val()

        const params = {
          id,
          modifierId,
          modifierType
        }
        const deleteCollectionResult = await deleteCollection(params)

        if (!deleteCollectionResult.success) {
          loading(false)
          toastError(deleteCollectionResult.message)
          return
        }

        await loadCollections()

        const $firstOption = $('select#combi_image_collection option:first-child'),
          firstOptionVal = parseInt($firstOption.val())

        resetImagesWrapper()

        if (!firstOptionVal) {
          disableImageSelection()
        } else {
          await loadImagesCollection($firstOption.val())
        }

        toastSuccess(deleteCollectionResult.message)
      } catch (e) {
        toastError(e.message)
      } finally {
        loading(false)
      }
    }

    /**
     * Handler for saving an image to a collection
     *
     * @param path
     * @returns {void}
     */
    const handleAddImageToCollection = async ({ url, localPath }) => {
      const params = { id: 0, src: url, localPath }
      const $image = $(imageBlockTemplate(params))

      appendImageToCollection($image)
      handleEditImageModal(params)
    }

    /**
     * Handler for open the Edit image modal button
     *
     * @param id
     * @param src
     * @param localPath
     * @param titles
     * @param altTitles
     */
    const handleEditImageModal = ({ id, src, localPath, titles = [], altTitles = [] }) => {
      $editModal.find('#collection-image-src').attr('src', src)
      $editModal.find('input[name=id]').val(id)
      $editModal.find('input[name=localPath]').val(localPath)
      $editModal.find('input[name=listId]').val(getSelectedList().id)

      titles.forEach(element => {
        $(`#image-title-${element.languageCode.toLowerCase()}`).val(element.value)
      })

      altTitles.forEach(element => {
        $(`#image-alt-${element.languageCode.toLowerCase()}`).val(element.value)
      })

      $editModal.fadeIn(() => focusFirstInputText($editModal))
    }

    /**
     * Handler for open the Edit image modal button
     */
    const handleEditCollectionModal = () => {
      $editCollectionModal.find('input[name=name]').val(getSelectedList().name)
      $editCollectionModal.find('input[name=id]').val(getSelectedList().id)
      // Ajax request
      $editCollectionModal.fadeIn(() => focusFirstInputText($editCollectionModal))
    }

    /**
     * Handles image list on change event
     *
     * @param val
     * @returns {Promise<void>}
     */
    const handleOnChangeImageCollection = async val => {
      loading(true)

      try {
        if (!val) {
          disableImageSelection()
          return
        }

        enableImageSelection()
        resetImagesWrapper()
        await loadImagesCollection(val)
      } catch (e) {
        toastError(e.message)
      } finally {
        loading(false)
      }
    }

    const focusFirstInputText = $element => {
      $element.find('input[type=text]:first').focus();
    }

    /**
     * Creates the single image wrapper
     *
     * @param data
     * @returns {string}
     */
    const imageBlockTemplate = data => {
      const { src = '//placehold.it/100x100', id } = data,
            mainImageText = jse.core.lang.translate('MAIN_IMAGE_LABEL', 'product_image_lists')

      return `<div class="collection-image-wrapper" data-list-element-id="${id}">\n` +
        '    <div class="actions">\n' +
        '        <a class="btn btn-primary move-image sort-handle" href="javascript:;">\n' +
        '            <i class="fa fa-arrows"></i>\n' +
        '        </a>\n' +
        '        <a class="btn btn-primary edit-image" href="javascript:;">\n' +
        '            <i class="fa fa-pencil"></i>\n' +
        '        </a>\n' +
        '        <a class="btn btn-danger delete-image" href="javascript:;">\n' +
        '            <i class="fa fa-trash"></i>\n' +
        '        </a>\n' +
        '    </div>\n' +
        `    <span class="main-image">${mainImageText}</span>\n` +
        `    <img alt="alt img" class="collection-image" src="${src}">\n` +
        '</div>'
    }

    /**
     * Gets the selected collection list
     *
     * @returns {{name: string, id: string}}
     */
    const getSelectedList = () => {
      const $selectedOption = $('#combi_image_collection option:selected')
      return {
        id: $selectedOption.val(),
        name: $selectedOption.html()
      }
    }

    /**
     * Bind events
     */
    const addEventListeners = () => {
      // Add event listeners
      addResponsiveFileManagerEventListener()

      createNewCollectionButtonEventListener()
      selectCollectionButtonEventListener()
      closeEditModalButtonEventListener()
      openEditImageModalEventListener()
      deleteImageButtonEventListener()
      imageCollectionOnChangeEventListener()
      addImageButtonEventListener()

      createCollectionButtonEventListener()
      deleteCollectionButtonEventListener()

      // Modal events
      preventSubmitFormModals()
      saveEditModalEventListener()
      cancelEditModalEventListener()

      closeEditCollectionModalButtonEventListener()
      openEditCollectionModalEventListener()
      cancelEditCollectionModalButtonEventListener()
      updateEditCollectionModalButtonEventListener()
    }

    /**
     * Toast "plugin"
     *
     * @param message
     * @param type
     */
    const toast = (message, type) => {
      const className = `status-${type}`

      $toastMessage
        .html(message)
        .removeClass((index, className) => (className.match(/(^|\s)status-\S+/g) || []).join(' '))
        .addClass(className)
        .stop()
        .fadeIn()

      clearTimeout(toastTimeoutID)

      toastTimeoutID = setTimeout(() => {
        $toastMessage.fadeOut(() => {
          $(this).removeClass(className)
        })
      }, 3000)
    }

    /**
     * Shows success toast
     *
     * @param message
     */
    const toastSuccess = message => toast(message, 'success')

    /**
     * Shows error toast
     *
     * @param message
     */
    const toastError = message => toast(message, 'error')

    /**
     * Handles loading spinner
     *
     * @param isLoading
     */
    const loading = isLoading => {
      if (isLoading) {
        loadingSpinner = jse.libs.loading_spinner.show($('#product-image-list-wrapper'), 9999)
        return
      }

      jse.libs.loading_spinner.hide(loadingSpinner)
      loadingSpinner = null
    }

    /**
     * Initialize the module
     *
     * @param done
     * @returns {void}
     */
    module.init = async (done) => {
      addEventListeners()

      $sortableList
        .sortable(sortableOptions)
        .on('sortupdate', handleOnSortUpdate)
        .disableSelection()

      loading(true)

      try {
        // Get image collections list
        await loadCollections(initialSelectedList)

        // Get images related with the current collection
        let selectedValue = initialSelectedList ?
          initialSelectedList :
          $('select#combi_image_collection option:selected').val()

        selectedValue = parseInt(selectedValue)

        if (!isNaN(selectedValue) && selectedValue > 0) {
          await loadImagesCollection(selectedValue)
        } else {
          disableImageSelection()
        }
      } catch (e) {
        toastError(e.message)
      } finally {
        loading(false)
        done()
      }
    }

    return module
  }
)